package org.openapi4j.operation.validator.util.convert;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.type.MapType;
import org.apache.commons.fileupload2.*;
import org.apache.commons.fileupload2.core.FileItemInput;
import org.apache.commons.fileupload2.core.FileItemInputIterator;
import org.apache.commons.fileupload2.core.FileUploadException;
import org.apache.commons.fileupload2.jakarta.JakartaServletFileUpload;
import org.openapi4j.core.model.OAIContext;
import org.openapi4j.core.util.IOUtil;
import org.openapi4j.core.util.TreeUtil;
import org.openapi4j.parser.model.v3.EncodingProperty;
import org.openapi4j.parser.model.v3.MediaType;
import org.openapi4j.parser.model.v3.Schema;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.function.LongSupplier;

import static org.openapi4j.core.model.v3.OAI3SchemaKeywords.*;

class MultipartConverter {
  private static final MultipartConverter INSTANCE = new MultipartConverter();

  private static final MapType MAP_TYPE = TreeUtil.json.getTypeFactory().constructMapType(
    HashMap.class,
    TreeUtil.json.getTypeFactory().constructType(String.class),
    TreeUtil.json.getTypeFactory().constructType(Object.class));

  private MultipartConverter() {
  }

  public static MultipartConverter instance() {
    return INSTANCE;
  }

  JsonNode convert(final OAIContext context, final MediaType mediaType, final String body, final String rawContentType, final String encoding) throws IOException {
    InputStream is = new ByteArrayInputStream(body.getBytes(encoding));
    return convert(context, mediaType, is, rawContentType, encoding);
  }

  JsonNode convert(final OAIContext context, final MediaType mediaType, final InputStream body, final String rawContentType, final String encoding) throws IOException {
	  
	  MCContext requestContext = new MCContext(body, rawContentType, encoding);
	  
    ObjectNode result = JsonNodeFactory.instance.objectNode();

    boolean multipartOptimization = (mediaType!=null && context!=null) ? context.isMultipartOptimization() : false;
    //System.out.println("multipartOptimization["+multipartOptimization+"]");

    java.util.List<String> notBinaries = new java.util.ArrayList();
    java.util.List<String> binaries = null;
    java.util.List<String> base64 = null;
    if(multipartOptimization){
        //System.out.println("ANALIZZO '"+mediaType.getSchema().getClass().getName()+"' CONTEXT["+context.getClass().getName()+"]");
        Map<String, Schema> map = mediaType.getSchema()!=null ? mediaType.getSchema().getProperties() : null;
        if(map!=null && !map.isEmpty()){
            binaries = new java.util.ArrayList();
            base64 = new java.util.ArrayList();
            for(String name: map.keySet()){
                //System.out.println("p["+name+"]");
                Schema propSchema = mediaType.getSchema().getProperty(name);
                if(propSchema!=null){
                    Schema flatSchema = propSchema.getFlatSchema(context);
                    if(flatSchema!=null){
                        if(org.openapi4j.core.model.v3.OAI3SchemaKeywords.TYPE_ARRAY.equals(flatSchema.getSupposedType(context))){
                            Schema arraySchema=flatSchema.getItemsSchema();
                            if(arraySchema!=null){
                                flatSchema = arraySchema.getFlatSchema(context);
                            }
                        }
                    }
                    if(flatSchema!=null){
                        if(org.openapi4j.core.model.v3.OAI3SchemaKeywords.FORMAT_BINARY.equals(flatSchema.getFormat())){
                            binaries.add(name);
                            continue;
                        }
                        else if(org.openapi4j.core.model.v3.OAI3SchemaKeywords.FORMAT_BASE64.equals(flatSchema.getFormat())){
                            base64.add(name);
                            continue;
                        }
                    }
                }
                notBinaries.add(name);
            }
        }
    }
    else{
        notBinaries.add("OptimizationDisabled");
    }

    //System.out.println("BASE64["+base64+"]");
    //System.out.println("BINARY["+binaries+"]");
    //System.out.println("NOT_BINARY["+notBinaries+"]");


    try {
      int unknown = 1;
      FileItemInputIterator iterator = new JakartaServletFileUpload<>().getItemIterator(requestContext);
      while (!notBinaries.isEmpty() && iterator.hasNext()) {
    	  FileItemInput item = iterator.next();
        String name = item.getFieldName();

        // Fix: attachments che non presentano nel valore dell'header Content-Disposition il 'filename' rientrano in questo if, altrimenti vanno nell'else
        //      però se per un bug del client, si entra in questo if si manda in crisi tutto poichè si cerca di costruire un json node da un pdf.
        //      il sistema va in sovraccarico (es. eclipse muore)

        String fileName = item.getName();
        boolean isBin = fileName!=null;
        Schema propSchema = null;
        Schema flatSchema = null;
        if (name!=null){
            propSchema = mediaType.getSchema().getProperty(name);
            if(propSchema!=null){
                flatSchema = propSchema.getFlatSchema(context);
		if(flatSchema!=null){
			if(org.openapi4j.core.model.v3.OAI3SchemaKeywords.TYPE_ARRAY.equals(flatSchema.getSupposedType(context))){
				Schema arraySchema=flatSchema.getItemsSchema();
				if(arraySchema!=null){
					 flatSchema = arraySchema.getFlatSchema(context);
				}
			}
                }
                if(flatSchema!=null){
                    if(org.openapi4j.core.model.v3.OAI3SchemaKeywords.FORMAT_BASE64.equals(flatSchema.getFormat()) ||
			org.openapi4j.core.model.v3.OAI3SchemaKeywords.FORMAT_BINARY.equals(flatSchema.getFormat())){
                       isBin=true;
                    }
		    else{
                       isBin=false;
		    }
                }
            }
        }

        //if (name!=null && item.isFormField()) {
        // isFormField si basa sulla presenza del parametro filename
        if (name!=null && !isBin ) {

          JsonNode convertedValue = mapValue(context, result, mediaType, item, name, encoding);
          if (convertedValue != null) {
            addValue(result, name, convertedValue);
          }
        } else { // Add file name only
          
           if(name==null){
		// Fix: parametro senza nome non rilevato
		name= "_unnamedParameter"+unknown;
		unknown++;
		
           }

	   // Fix: fileName è opzionale
	   String content = fileName;
	   if(fileName==null){
		content=name;
	   }  

	   // Fix: per poter utilizzare il validatore base64 in org.openapi4j.schema.validator.v3.FormatValidator
           if(flatSchema!=null && content!=null){
                 if(org.openapi4j.core.model.v3.OAI3SchemaKeywords.FORMAT_BASE64.equals(flatSchema.getFormat())){
		    content = org.apache.commons.codec.binary.Base64.encodeBase64String(content.getBytes());
		 }
           }

           addValue(result, name, JsonNodeFactory.instance.textNode(content));
        }

        if(multipartOptimization){
            if(notBinaries.contains(name)){
                notBinaries.remove(name);
            } else if(binaries.contains(name)){
                binaries.remove(name);
            } else if(base64.contains(name)){
                base64.remove(name);
            }

            //System.out.println("Dopo aver analizzato '"+name+"':");
            //System.out.println("   BASE64["+base64+"]");
            //System.out.println("   BINARY["+binaries+"]");
            //System.out.println("   NOT_BINARY["+notBinaries+"]");
        }

      }

    if(multipartOptimization){
        //System.out.println("GESTIONE FINALE");
        if(!binaries.isEmpty()){
            for(String n: binaries){
                //System.out.println("ADD BIN '"+n+"'");
                addValue(result, n, JsonNodeFactory.instance.textNode(n));
            }
        }
        if(!base64.isEmpty()){
            for(String n: base64){
                String content = org.apache.commons.codec.binary.Base64.encodeBase64String(n.getBytes());
                //System.out.println("ADD BASE64 '"+n+"': "+content);
                addValue(result, n, JsonNodeFactory.instance.textNode(content));
            }
        }
    }

    } catch (FileUploadException ex) {
      throw new IOException(ex);
    }

    return result;
  }

  private JsonNode mapValue(OAIContext context, ObjectNode result, MediaType mediaType, FileItemInput item, String name, String encoding) throws IOException {
    Schema propSchema = mediaType.getSchema().getProperty(name);
    String itemContentType = item.getContentType();

    if (itemContentType != null) {
      final int checkResult = checkContentType(context, propSchema, mediaType.getEncoding(name), item);
      if (checkResult == -1) {
        // content type mismatch
        String content = IOUtil.toString(item.getInputStream(), encoding);
        return JsonNodeFactory.instance.textNode(content);
      } else if (checkResult == 0) {
        // Process with the given content type
        String content = IOUtil.toString(item.getInputStream(), encoding);
        try {
          return ContentConverter.convert(context, new MediaType().setSchema(propSchema), itemContentType, null, content);
        } catch (IOException ex) {
          // content type mismatch
          return JsonNodeFactory.instance.textNode(content);
        }
      }
    }

    // Process as JSON
    return convertToJsonNode(context, result, name, propSchema, item, encoding);
  }

  /**
   * Check content type.
   *
   * @return -1: in case of mismatch<br/>
   * 0: if content should be processed with the given content type<br/>
   * 1: if the content should be processed as JSON.<br/>
   */
  private int checkContentType(OAIContext context, Schema propSchema, EncodingProperty encProperty, FileItemInput item) {
    String itemContentType = item.getContentType();
    String specContentType = (encProperty != null && encProperty.getContentType() != null) ? encProperty.getContentType() : null;

    // fix null pointer
    if(propSchema==null){
       return -1;
    }

    // Check given content type against spec content type
    if (specContentType != null && !itemContentType.equals(specContentType)) {
      return -1;
    }

    // Cheking by default value
    Schema flatSchema = propSchema.getFlatSchema(context);
    switch (flatSchema.getSupposedType(context)) {
      case TYPE_OBJECT:
        // for object - application/json
        return itemContentType.equals("application/json") ? 1 : 0;
      case TYPE_ARRAY:
        // for array - defined based on the inner type
        return checkContentType(context, flatSchema.getItemsSchema(), encProperty, item);
      case TYPE_STRING:
        // for string with format being binary - application/octet-stream
        if (FORMAT_BINARY.equals(flatSchema.getFormat())) {
          return itemContentType.equals("application/octet-stream") ? 1 : 0;
        }
      default:
        // for other primitive types - text/plain
        return itemContentType.equals("text/plain") ? 1 : 0;
    }
  }

  private JsonNode convertToJsonNode(final OAIContext context,
                                     final ObjectNode result,
                                     final String name,
                                     final Schema schema,
                                     final FileItemInput item,
                                     final String encoding) throws IOException {

    if (schema == null) {
      return TypeConverter.instance().convertPrimitive(context, null, IOUtil.toString(item.getInputStream(), encoding));
    }

    String supposedType = schema.getSupposedType(context);
    if(supposedType==null){
        return TypeConverter.instance().convertPrimitive(context, schema, IOUtil.toString(item.getInputStream(), encoding));
    }

    switch (supposedType) {
      case TYPE_OBJECT:
        Map<String, Object> jsonContent = TreeUtil.json.readValue(item.getInputStream(), MAP_TYPE);
        // FIX!
        //return TypeConverter.instance().convertObject(context, schema, jsonContent);
        try{
           return org.openapi4j.core.util.TreeUtil.toJsonNode(jsonContent);
        }catch(Exception e){
           throw new IOException(e.getMessage(),e);
        }
      case TYPE_ARRAY:
        // Special case for arrays
        // They can be referenced multiple times in different ways

        Schema schemaItems = null;
        Schema flatSchema = schema.getFlatSchema(context);
        if(flatSchema!=null){
             schemaItems = flatSchema.getItemsSchema();
        }else{
             schemaItems = schema.getItemsSchema();
        }
        if(schemaItems!=null){
            flatSchema = schemaItems.getFlatSchema(context);
            if(flatSchema!=null){
                schemaItems = flatSchema;
            }
        }

        JsonNode convertedValue = convertToJsonNode(context, result, name, schemaItems, item, encoding);
        JsonNode previousValue = result.get(name);
        if ((previousValue instanceof ArrayNode)) {
          ((ArrayNode) previousValue).add(convertedValue);
        } else {
          result.set(name, JsonNodeFactory.instance.arrayNode().add(convertedValue));
        }
        return null;
      default:
        return TypeConverter.instance().convertPrimitive(context, schema, IOUtil.toString(item.getInputStream(), encoding));
    }
  }

  private void addValue(ObjectNode result, String name, JsonNode value) {
    // Check if value is already referenced
    // If so, add new value to an array
    JsonNode previousValue = result.get(name);
    if (previousValue != null) {
      if (previousValue instanceof ArrayNode) {
        ((ArrayNode) previousValue).add(value);
      } else {
        ArrayNode values = JsonNodeFactory.instance.arrayNode();
        values.add(previousValue);
        values.add(value);

        result.set(name, values);
      }
    } else {
      result.set(name, value);
    }
  }

}

class MCContext extends org.apache.commons.fileupload2.core.AbstractRequestContext<String>{

	String encoding;
	String contentType;
	InputStream body;
	
	public MCContext(InputStream body, String contentType, String encoding){
		super(MCContextUtils::length, new LongSupplier() {
			
			@Override
			public long getAsLong() {
				return 0;
			}
		}, "NOP");
		this.encoding = encoding;
		this.contentType = contentType;
		this.body = body;
	}
	
	@Override
	public String getCharacterEncoding() {
		return this.encoding;
	}

	@Override
	public String getContentType() {
		return this.contentType;
	}

	@Override
	public InputStream getInputStream() throws IOException {
		return this.body;
	}
	
}

class MCContextUtils {
   public static String length(String x) { return x.length()+""; }
}
