/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection;

import java.net.InetSocketAddress;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import org.redisson.api.NodeType;
import org.redisson.api.RFuture;
import org.redisson.client.RedisClient;
import org.redisson.client.RedisConnection;
import org.redisson.client.RedisConnectionException;
import org.redisson.client.RedisPubSubConnection;
import org.redisson.client.protocol.RedisCommand;
import org.redisson.client.protocol.RedisCommands;
import org.redisson.config.MasterSlaveServersConfig;
import org.redisson.config.ReadMode;
import org.redisson.config.SubscriptionMode;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.pool.MasterConnectionPool;
import org.redisson.connection.pool.MasterPubSubConnectionPool;
import org.redisson.connection.pool.PubSubConnectionPool;
import org.redisson.connection.pool.SlaveConnectionPool;
import org.redisson.misc.RedisURI;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MasterSlaveEntry {
    final Logger log = LoggerFactory.getLogger(this.getClass());
    volatile ClientConnectionsEntry masterEntry;
    int references;
    final MasterSlaveServersConfig config;
    final ConnectionManager connectionManager;
    final MasterConnectionPool masterConnectionPool;
    final MasterPubSubConnectionPool masterPubSubConnectionPool;
    final PubSubConnectionPool slavePubSubConnectionPool;
    final SlaveConnectionPool slaveConnectionPool;
    final Map<RedisClient, ClientConnectionsEntry> client2Entry = new ConcurrentHashMap<RedisClient, ClientConnectionsEntry>();
    final AtomicBoolean active = new AtomicBoolean(true);
    final AtomicBoolean noPubSubSlaves = new AtomicBoolean();
    volatile int availableSlaves = -1;
    volatile boolean aofEnabled;

    public MasterSlaveEntry(ConnectionManager connectionManager, MasterSlaveServersConfig config) {
        this.connectionManager = connectionManager;
        this.config = config;
        this.slaveConnectionPool = new SlaveConnectionPool(config, connectionManager, this);
        this.slavePubSubConnectionPool = new PubSubConnectionPool(config, connectionManager, this);
        this.masterConnectionPool = new MasterConnectionPool(config, connectionManager, this);
        this.masterPubSubConnectionPool = new MasterPubSubConnectionPool(config, connectionManager, this);
    }

    public MasterSlaveServersConfig getConfig() {
        return this.config;
    }

    public CompletableFuture<Void> initSlaveBalancer(Function<RedisURI, String> hostnameMapper) {
        ArrayList<CompletableFuture<Void>> result = new ArrayList<CompletableFuture<Void>>(this.config.getSlaveAddresses().size());
        for (String address : this.config.getSlaveAddresses()) {
            RedisURI uri = new RedisURI(address);
            String hostname = hostnameMapper.apply(uri);
            CompletableFuture<Void> f = this.addSlave(uri, hostname);
            result.add(f);
        }
        CompletableFuture<Void> future = CompletableFuture.allOf(result.toArray(new CompletableFuture[0]));
        return future.thenAccept(v -> this.useMasterAsSlave());
    }

    private void useMasterAsSlave() {
        if (this.hasNoSlaves() || this.config.getReadMode() == ReadMode.MASTER_SLAVE) {
            this.addSlaveEntry(this.masterEntry);
        } else {
            this.removeSlaveEntry(this.masterEntry);
        }
    }

    private void removeSlaveEntry(ClientConnectionsEntry entry) {
        this.client2Entry.remove(entry.getClient());
        if (this.config.getSubscriptionMode() == SubscriptionMode.SLAVE) {
            entry.reattachPubSub();
        }
    }

    private void addSlaveEntry(ClientConnectionsEntry entry) {
        this.client2Entry.putIfAbsent(entry.getClient(), entry);
    }

    private boolean hasNoSlaves() {
        int count = (int)this.client2Entry.values().stream().filter(e -> !e.isFreezed() && e.getNodeType() == NodeType.SLAVE).count();
        return count == 0;
    }

    public CompletableFuture<RedisClient> setupMasterEntry(InetSocketAddress address, RedisURI uri) {
        RedisClient client = this.connectionManager.createClient(NodeType.MASTER, address, uri, null);
        return this.setupMasterEntry(client);
    }

    public CompletableFuture<RedisClient> setupMasterEntry(RedisURI address) {
        return this.setupMasterEntry(address, null);
    }

    public CompletableFuture<RedisClient> setupMasterEntry(RedisURI address, String sslHostname) {
        RedisClient client = this.connectionManager.createClient(NodeType.MASTER, address, sslHostname);
        return this.setupMasterEntry(client);
    }

    private CompletableFuture<RedisClient> setupMasterEntry(RedisClient client) {
        CompletableFuture<InetSocketAddress> addrFuture = client.resolveAddr();
        return ((CompletableFuture)addrFuture.thenCompose(res -> {
            ClientConnectionsEntry entry = new ClientConnectionsEntry(client, this.config.getMasterConnectionMinimumIdleSize(), this.config.getMasterConnectionPoolSize(), this.connectionManager, NodeType.MASTER, this.config);
            ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
            CompletableFuture<Void> writeFuture = entry.initConnections(this.config.getMasterConnectionMinimumIdleSize());
            futures.add(writeFuture);
            if (this.config.getSubscriptionMode() == SubscriptionMode.MASTER) {
                CompletableFuture<Void> pubSubFuture = entry.initPubSubConnections(this.config.getSubscriptionConnectionMinimumIdleSize());
                futures.add(pubSubFuture);
            }
            return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).thenApply(r -> {
                this.masterEntry = entry;
                if (!this.config.isSlaveNotUsed()) {
                    this.addSlaveEntry(this.masterEntry);
                }
                return client;
            });
        })).whenComplete((r, e) -> {
            if (e != null) {
                client.shutdownAsync();
            }
        });
    }

    public boolean slaveDown(InetSocketAddress address) {
        ClientConnectionsEntry connectionEntry = this.getEntry(address);
        ClientConnectionsEntry entry = this.freeze(connectionEntry, ClientConnectionsEntry.FreezeReason.MANAGER);
        if (entry == null) {
            return false;
        }
        return this.slaveDown(entry);
    }

    public boolean slaveDown(RedisURI address) {
        ClientConnectionsEntry connectionEntry = this.getEntry(address);
        ClientConnectionsEntry entry = this.freeze(connectionEntry, ClientConnectionsEntry.FreezeReason.MANAGER);
        if (entry == null) {
            return false;
        }
        return this.slaveDown(entry);
    }

    private boolean slaveDown(ClientConnectionsEntry entry) {
        if (!this.config.isSlaveNotUsed() && !this.masterEntry.getClient().getAddr().equals(entry.getClient().getAddr()) && this.hasNoSlaves()) {
            this.addSlaveEntry(this.masterEntry);
            this.log.info("master {} is used as slave", (Object)this.masterEntry.getClient().getAddr());
        }
        entry.nodeDown();
        return true;
    }

    public void shutdownAndReconnectAsync(RedisClient client, Throwable cause) {
        ClientConnectionsEntry entry = this.getEntry(client);
        if (this.slaveDown(entry, ClientConnectionsEntry.FreezeReason.RECONNECT)) {
            this.log.error("Redis node {} has been disconnected", (Object)entry.getClient().getAddr(), (Object)cause);
            this.scheduleCheck(entry);
        }
    }

    private void scheduleCheck(ClientConnectionsEntry entry) {
        this.connectionManager.getServiceManager().newTimeout(timeout -> {
            boolean res = entry.getLock().execute(() -> {
                if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT || this.connectionManager.getServiceManager().isShuttingDown()) {
                    return false;
                }
                return true;
            });
            if (!res) {
                return;
            }
            RFuture<RedisConnection> connectionFuture = entry.getClient().connectAsync();
            connectionFuture.whenComplete((c, e) -> {
                boolean res2 = entry.getLock().execute(() -> {
                    if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT) {
                        return false;
                    }
                    return true;
                });
                if (!res2) {
                    return;
                }
                if (e != null) {
                    this.scheduleCheck(entry);
                    return;
                }
                if (!c.isActive()) {
                    c.closeAsync();
                    this.scheduleCheck(entry);
                    return;
                }
                RFuture f = c.async(RedisCommands.PING, new Object[0]);
                f.whenComplete((t, ex) -> {
                    try {
                        boolean res3 = entry.getLock().execute(() -> {
                            if (entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT) {
                                return false;
                            }
                            return true;
                        });
                        if (!res3) {
                            return;
                        }
                        if ("PONG".equals(t)) {
                            CompletableFuture<Boolean> ff = this.slaveUpAsync(entry);
                            ff.thenAccept(r -> {
                                if (r.booleanValue()) {
                                    this.log.info("slave {} has been successfully reconnected", (Object)entry.getClient().getAddr());
                                }
                            });
                        } else {
                            this.scheduleCheck(entry);
                        }
                    }
                    finally {
                        c.closeAsync();
                    }
                });
            });
        }, this.config.getFailedSlaveReconnectionInterval(), TimeUnit.MILLISECONDS);
    }

    private boolean slaveDown(ClientConnectionsEntry entry, ClientConnectionsEntry.FreezeReason freezeReason) {
        ClientConnectionsEntry e = this.freeze(entry, freezeReason);
        if (e == null) {
            return false;
        }
        return this.slaveDown(entry);
    }

    public void masterDown() {
        this.masterEntry.nodeDown();
    }

    public boolean hasSlave(RedisClient redisClient) {
        return this.getEntry(redisClient) != null;
    }

    public boolean hasSlave(InetSocketAddress addr) {
        return this.getEntry(addr) != null;
    }

    public boolean hasSlave(RedisURI addr) {
        return this.getEntry(addr) != null;
    }

    public CompletableFuture<Void> addSlave(RedisURI address) {
        return this.addSlave(address, null);
    }

    public CompletableFuture<Void> addSlave(InetSocketAddress address, RedisURI uri) {
        return this.addSlave(address, uri, null);
    }

    public CompletableFuture<Void> addSlave(RedisClient client) {
        this.noPubSubSlaves.set(false);
        CompletableFuture<InetSocketAddress> addrFuture = client.resolveAddr();
        return ((CompletableFuture)addrFuture.thenCompose(res -> {
            ClientConnectionsEntry entry = new ClientConnectionsEntry(client, this.config.getSlaveConnectionMinimumIdleSize(), this.config.getSlaveConnectionPoolSize(), this.connectionManager, NodeType.SLAVE, this.config);
            CompletableFuture<Void> slaveFuture = entry.initConnections(this.config.getSlaveConnectionMinimumIdleSize());
            CompletableFuture<Void> pubSubFuture = entry.initPubSubConnections(this.config.getSubscriptionConnectionMinimumIdleSize());
            return CompletableFuture.allOf(slaveFuture, pubSubFuture).thenAccept(r -> this.addSlaveEntry(entry));
        })).whenComplete((r, ex) -> {
            if (ex != null) {
                client.shutdownAsync();
            }
        });
    }

    public CompletableFuture<Void> addSlave(InetSocketAddress address, RedisURI uri, String sslHostname) {
        RedisClient client = this.connectionManager.createClient(NodeType.SLAVE, address, uri, sslHostname);
        return this.addSlave(client);
    }

    public CompletableFuture<Void> addSlave(RedisURI address, String sslHostname) {
        RedisClient client = this.connectionManager.createClient(NodeType.SLAVE, address, sslHostname);
        return this.addSlave(client);
    }

    public Collection<ClientConnectionsEntry> getAllEntries() {
        return Collections.unmodifiableCollection(this.client2Entry.values());
    }

    public ClientConnectionsEntry getEntry(InetSocketAddress address) {
        InetSocketAddress masterAddr = this.masterEntry.getClient().getAddr();
        if (masterAddr.getAddress().equals(address.getAddress()) && masterAddr.getPort() == address.getPort()) {
            return this.masterEntry;
        }
        for (ClientConnectionsEntry entry : this.client2Entry.values()) {
            InetSocketAddress addr = entry.getClient().getAddr();
            if (!addr.getAddress().equals(address.getAddress()) || addr.getPort() != address.getPort()) continue;
            return entry;
        }
        return null;
    }

    public ClientConnectionsEntry getEntry(RedisClient redisClient) {
        if (this.masterEntry.getClient().equals(redisClient)) {
            return this.masterEntry;
        }
        return this.client2Entry.get(redisClient);
    }

    public ClientConnectionsEntry getEntry(RedisURI addr) {
        if (addr.equals(this.masterEntry.getClient().getAddr())) {
            return this.masterEntry;
        }
        for (ClientConnectionsEntry entry : this.client2Entry.values()) {
            InetSocketAddress entryAddr = entry.getClient().getAddr();
            if (!addr.equals(entryAddr)) continue;
            return entry;
        }
        return null;
    }

    public boolean isInit() {
        return this.masterEntry != null;
    }

    public RedisClient getClient() {
        return this.masterEntry.getClient();
    }

    private CompletableFuture<Boolean> slaveUpAsync(ClientConnectionsEntry entry) {
        this.noPubSubSlaves.set(false);
        CompletableFuture<Boolean> f = this.unfreezeAsync(entry, ClientConnectionsEntry.FreezeReason.RECONNECT);
        return f.thenApply(r -> {
            if (r.booleanValue()) {
                this.excludeMasterFromSlaves(entry.getClient().getAddr());
                return r;
            }
            return r;
        });
    }

    public CompletableFuture<Boolean> slaveUpAsync(RedisURI address) {
        this.noPubSubSlaves.set(false);
        CompletableFuture<Boolean> f = this.unfreezeAsync(address);
        return f.thenApply(r -> {
            if (r.booleanValue()) {
                this.excludeMasterFromSlaves(address);
                return r;
            }
            return r;
        });
    }

    public boolean excludeMasterFromSlaves(RedisURI address) {
        InetSocketAddress addr = this.masterEntry.getClient().getAddr();
        if (address.equals(addr) || this.config.getReadMode() == ReadMode.MASTER_SLAVE) {
            return false;
        }
        this.removeSlaveEntry(this.masterEntry);
        this.log.info("master {} excluded from slaves", (Object)addr);
        return true;
    }

    public boolean excludeMasterFromSlaves(InetSocketAddress address) {
        InetSocketAddress addr = this.masterEntry.getClient().getAddr();
        if (this.config.isSlaveNotUsed() || addr.equals(address) || this.config.getReadMode() == ReadMode.MASTER_SLAVE) {
            return false;
        }
        this.removeSlaveEntry(this.masterEntry);
        this.log.info("master {} excluded from slaves", (Object)addr);
        return true;
    }

    public CompletableFuture<Boolean> slaveUpNoMasterExclusionAsync(RedisURI address) {
        this.noPubSubSlaves.set(false);
        return this.unfreezeAsync(address);
    }

    public CompletableFuture<Boolean> slaveUpNoMasterExclusionAsync(InetSocketAddress address) {
        this.noPubSubSlaves.set(false);
        return this.unfreezeAsync(address);
    }

    public CompletableFuture<Boolean> slaveUpAsync(InetSocketAddress address) {
        this.noPubSubSlaves.set(false);
        CompletableFuture<Boolean> f = this.unfreezeAsync(address);
        return f.thenApply(r -> {
            if (r.booleanValue()) {
                this.excludeMasterFromSlaves(address);
                return r;
            }
            return r;
        });
    }

    public CompletableFuture<RedisClient> changeMaster(RedisURI address) {
        ClientConnectionsEntry oldMaster = this.masterEntry;
        CompletableFuture<RedisClient> future = this.setupMasterEntry(address);
        return this.changeMaster(address, oldMaster, future);
    }

    public CompletableFuture<RedisClient> changeMaster(InetSocketAddress address, RedisURI uri) {
        ClientConnectionsEntry oldMaster = this.masterEntry;
        CompletableFuture<RedisClient> future = this.setupMasterEntry(address, uri);
        return this.changeMaster(uri, oldMaster, future);
    }

    private CompletableFuture<RedisClient> changeMaster(RedisURI address, ClientConnectionsEntry oldMaster, CompletableFuture<RedisClient> future) {
        return future.whenComplete((newMasterClient, e) -> {
            if (e != null) {
                if (oldMaster != this.masterEntry) {
                    this.removeMaster(this.masterEntry);
                    this.masterEntry = oldMaster;
                }
                this.log.error("Unable to change master from: {} to: {}", new Object[]{oldMaster.getClient().getAddr(), address, e});
                return;
            }
            ClientConnectionsEntry entry = this.getAllEntries().stream().filter(node -> node.getNodeType() == NodeType.SLAVE && node.getClient().getAddr().equals(this.masterEntry.getClient().getAddr())).findAny().orElse(null);
            if (entry != null) {
                this.removeSlaveEntry(entry);
                entry.nodeDown();
                entry.shutdownAsync();
                this.log.info("new master {} excluded from slaves", (Object)this.masterEntry.getClient().getAddr());
            }
            this.removeMaster(oldMaster);
            if (!this.config.isSlaveNotUsed()) {
                this.useMasterAsSlave();
            }
            this.log.info("master {} has been changed to {}", (Object)oldMaster.getClient().getAddr(), (Object)this.masterEntry.getClient().getAddr());
        });
    }

    private void removeMaster(ClientConnectionsEntry masterEntry) {
        this.removeSlaveEntry(masterEntry);
        masterEntry.nodeDown();
        masterEntry.shutdownAsync();
    }

    public CompletableFuture<Void> shutdownAsync() {
        if (!this.active.compareAndSet(true, false)) {
            return CompletableFuture.completedFuture(null);
        }
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>(this.client2Entry.size() + 1);
        if (this.masterEntry != null) {
            futures.add(this.masterEntry.shutdownAsync());
        }
        for (ClientConnectionsEntry entry : this.client2Entry.values()) {
            futures.add(entry.shutdownAsync());
        }
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    public CompletableFuture<RedisConnection> connectionWriteOp(RedisCommand<?> command) {
        return this.masterConnectionPool.get(command, false);
    }

    public CompletableFuture<RedisConnection> trackedConnectionWriteOp(RedisCommand<?> command) {
        return this.masterConnectionPool.get(command, true);
    }

    public CompletableFuture<RedisConnection> redirectedConnectionWriteOp(RedisCommand<?> command, RedisURI addr) {
        return this.connectionReadOp(command, addr);
    }

    public CompletableFuture<RedisConnection> connectionReadOp(RedisCommand<?> command, boolean trackChanges) {
        if (this.config.getReadMode() == ReadMode.MASTER) {
            if (trackChanges) {
                return this.trackedConnectionWriteOp(command);
            }
            return this.connectionWriteOp(command);
        }
        return this.slaveConnectionPool.get((RedisCommand)command, trackChanges);
    }

    public CompletableFuture<RedisConnection> connectionReadOp(RedisCommand<?> command, RedisURI addr) {
        ClientConnectionsEntry entry = this.getEntry(addr);
        if (entry != null) {
            return this.slaveConnectionPool.get((RedisCommand)command, entry, false);
        }
        RedisConnectionException exception = new RedisConnectionException("Can't find entry for " + addr + " command " + command);
        CompletableFuture<RedisConnection> f = new CompletableFuture<RedisConnection>();
        f.completeExceptionally(exception);
        return f;
    }

    public CompletableFuture<RedisConnection> connectionReadOp(RedisCommand<?> command, RedisClient client, boolean trackChanges) {
        if (this.config.getReadMode() == ReadMode.MASTER) {
            if (trackChanges) {
                return this.trackedConnectionWriteOp(command);
            }
            return this.connectionWriteOp(command);
        }
        ClientConnectionsEntry entry = this.getEntry(client);
        if (entry != null) {
            return this.slaveConnectionPool.get((RedisCommand)command, entry, trackChanges);
        }
        RedisConnectionException exception = new RedisConnectionException("Can't find entry for " + client + " command " + command);
        CompletableFuture<RedisConnection> f = new CompletableFuture<RedisConnection>();
        f.completeExceptionally(exception);
        return f;
    }

    public CompletableFuture<RedisPubSubConnection> nextPubSubConnection(ClientConnectionsEntry entry) {
        if (entry != null) {
            return this.slavePubSubConnectionPool.get(entry);
        }
        if (this.config.getSubscriptionMode() == SubscriptionMode.MASTER) {
            return this.masterPubSubConnectionPool.get();
        }
        if (this.noPubSubSlaves.get()) {
            return this.masterPubSubConnectionPool.get();
        }
        CompletableFuture<RedisPubSubConnection> future = this.slavePubSubConnectionPool.get();
        return ((CompletableFuture)future.handle((r, e) -> {
            if (e != null) {
                if (this.noPubSubSlaves.compareAndSet(false, true)) {
                    this.log.warn("No slaves for master: {} PubSub connections established with the master node.", (Object)this.masterEntry.getClient().getAddr(), e);
                }
                return this.masterPubSubConnectionPool.get();
            }
            return CompletableFuture.completedFuture(r);
        })).thenCompose(f -> f);
    }

    public void returnPubSubConnection(RedisPubSubConnection connection) {
        ClientConnectionsEntry entry = this.getEntry(connection.getRedisClient());
        if (entry == null) {
            connection.closeAsync();
            return;
        }
        entry.returnConnection(connection);
    }

    public void releaseWrite(RedisConnection connection) {
        this.masterEntry.returnConnection(connection);
    }

    public void releaseRead(RedisConnection connection) {
        if (this.config.getReadMode() == ReadMode.MASTER) {
            this.releaseWrite(connection);
            return;
        }
        ClientConnectionsEntry entry = this.getEntry(connection.getRedisClient());
        if (entry == null) {
            connection.closeAsync();
            return;
        }
        entry.returnConnection(connection);
    }

    public void incReference() {
        ++this.references;
    }

    public int decReference() {
        return --this.references;
    }

    public int getReferences() {
        return this.references;
    }

    public String toString() {
        return "MasterSlaveEntry [masterEntry=" + this.masterEntry + "]";
    }

    private ClientConnectionsEntry freeze(ClientConnectionsEntry connectionEntry, ClientConnectionsEntry.FreezeReason freezeReason) {
        if (connectionEntry == null || connectionEntry.getClient().getConfig().getFailedNodeDetector().isNodeFailed() && connectionEntry.getFreezeReason() == ClientConnectionsEntry.FreezeReason.RECONNECT && freezeReason == ClientConnectionsEntry.FreezeReason.RECONNECT) {
            return null;
        }
        return connectionEntry.getLock().execute(() -> {
            if (connectionEntry.isFreezed()) {
                return null;
            }
            if (connectionEntry.getFreezeReason() == null || connectionEntry.getFreezeReason() == ClientConnectionsEntry.FreezeReason.RECONNECT || freezeReason == ClientConnectionsEntry.FreezeReason.MANAGER && connectionEntry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.MANAGER && connectionEntry.getNodeType() == NodeType.SLAVE) {
                connectionEntry.setFreezeReason(freezeReason);
                return connectionEntry;
            }
            return connectionEntry;
        });
    }

    private CompletableFuture<Boolean> unfreezeAsync(RedisURI address) {
        ClientConnectionsEntry entry = this.getEntry(address);
        if (entry == null) {
            this.log.error("Can't find {} in slaves! Available slaves: {}", (Object)address, this.client2Entry.keySet());
            return CompletableFuture.completedFuture(false);
        }
        return this.unfreezeAsync(entry, ClientConnectionsEntry.FreezeReason.MANAGER);
    }

    private CompletableFuture<Boolean> unfreezeAsync(InetSocketAddress address) {
        ClientConnectionsEntry entry = this.getEntry(address);
        if (entry == null) {
            this.log.error("Can't find {} in slaves! Available slaves: {}", (Object)address, this.client2Entry.keySet());
            return CompletableFuture.completedFuture(false);
        }
        return this.unfreezeAsync(entry, ClientConnectionsEntry.FreezeReason.MANAGER);
    }

    private CompletableFuture<Boolean> unfreezeAsync(ClientConnectionsEntry entry, ClientConnectionsEntry.FreezeReason freezeReason) {
        return this.unfreezeAsync(entry, freezeReason, 0);
    }

    private CompletableFuture<Boolean> unfreezeAsync(ClientConnectionsEntry entry, ClientConnectionsEntry.FreezeReason freezeReason, int retry) {
        return entry.getLock().execute(() -> {
            if (!entry.isFreezed()) {
                return CompletableFuture.completedFuture(false);
            }
            if (!(freezeReason == ClientConnectionsEntry.FreezeReason.RECONNECT && entry.getFreezeReason() != ClientConnectionsEntry.FreezeReason.RECONNECT || entry.isInitialized())) {
                entry.setInitialized(true);
                ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>(2);
                futures.add(entry.initConnections(this.config.getSlaveConnectionMinimumIdleSize()));
                futures.add(entry.initPubSubConnections(this.config.getSubscriptionConnectionMinimumIdleSize()));
                CompletableFuture<Void> future = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
                CompletableFuture f = new CompletableFuture();
                future.whenComplete((r, e) -> {
                    if (e != null) {
                        MasterSlaveServersConfig config = this.connectionManager.getServiceManager().getConfig();
                        int maxAttempts = config.getRetryAttempts();
                        this.log.error("Unable to unfreeze entry: {} attempt: {} of {}", new Object[]{entry, retry, maxAttempts, e});
                        entry.setInitialized(false);
                        if (retry < maxAttempts) {
                            Duration timeout = config.getRetryDelay().calcDelay(retry);
                            this.connectionManager.getServiceManager().newTimeout(t -> {
                                CompletableFuture<Boolean> ff = this.unfreezeAsync(entry, freezeReason, retry + 1);
                                this.connectionManager.getServiceManager().transfer(ff, f);
                            }, timeout.toMillis(), TimeUnit.MILLISECONDS);
                        } else {
                            f.complete(false);
                        }
                        return;
                    }
                    entry.getClient().getConfig().getFailedNodeDetector().onConnectSuccessful();
                    entry.setFreezeReason(null);
                    this.log.debug("Unfreezed entry: {} after {} attempts", (Object)entry, (Object)retry);
                    f.complete(true);
                });
                return f;
            }
            return CompletableFuture.completedFuture(false);
        });
    }

    public ClientConnectionsEntry getEntry() {
        return this.masterEntry;
    }

    public int getAvailableSlaves() {
        return this.availableSlaves;
    }

    public void setAvailableSlaves(int slaves) {
        this.availableSlaves = slaves;
    }

    public boolean isAofEnabled() {
        return this.aofEnabled;
    }

    public void setAofEnabled(boolean aof) {
        this.aofEnabled = aof;
    }
}

